/*
 * Driver for Vision Components GmbH VC MIPI IMX290 Sensor module - CMOS Image Sensor from Sony
 *
 * Copyright (C) 2019, Michael Steinel <steinel@vision-components.com>
 *
 * based on 
 *
 * IMX219 driver, Guennadi Liakhovetski <g.liakhovetski@gmx.de>
 *
 * Copyright (C) 2014, Andrew Chew <achew@nvidia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/of_graph.h>
#include <linux/slab.h>
#include <linux/videodev2.h>
#include <media/v4l2-ctrls.h>
#include <media/v4l2-device.h>
#include <media/v4l2-fwnode.h>
#include <media/v4l2-image-sizes.h>
#include <media/v4l2-mediabus.h>

/* IMX290 supported geometry */
#define IMX290_TABLE_END		0xffff

/* In dB*10 */
#define IMX290_DIGITAL_GAIN_MIN		0
#define IMX290_DIGITAL_GAIN_MAX		240
#define IMX290_DIGITAL_GAIN_DEFAULT	0

/* In usec */
#define IMX290_DIGITAL_EXPOSURE_MIN	29
#define IMX290_DIGITAL_EXPOSURE_MAX	7602176
#define IMX290_DIGITAL_EXPOSURE_DEFAULT	10000

/* VC Sensor Mode - default 10-Bit Streaming */
static int sensor_mode = 0;
module_param(sensor_mode, int, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
MODULE_PARM_DESC(sensor_mode, "VC Sensor Mode: 0=10bit_stream 1=10bit_4lanes");

struct imx290_reg {
	u16 addr;
	u8 val;
};

struct imx290_mode {
	u32 sensor_mode;
	u32 sensor_depth;
	u32 sensor_ext_trig;
	u32 width;
	u32 height;
	u32 max_fps;
	const struct imx290_reg *reg_list;
};

/* MCLK:54MHz  1920x1080  60fps  MIPI LANES 2 */
static const struct imx290_reg imx290_init_tab_1920_1080_60fps[] = {
	{IMX290_TABLE_END, 0x00}
};

/* MCLK:54MHz  1920x1080  120fps  MIPI LANES 4 */
static const struct imx290_reg imx290_init_tab_1920_1080_120fps[] = {
	{IMX290_TABLE_END, 0x00}
};

static const struct imx290_reg start[] = {
	{0x3000, 0x00},		/* mode select streaming on */
	{0x3002, 0x00},		/* mode select streaming on */
	{IMX290_TABLE_END, 0x00}
};

static const struct imx290_reg stop[] = {
	{0x3002, 0x01},		/* mode select streaming off */
	{0x3000, 0x01},		/* mode select streaming off */
	{IMX290_TABLE_END, 0x00}
};


#define SIZEOF_I2C_TRANSBUF 32

struct vc_rom_table {
	char magic[12];
	char manuf[32];
	u16 manuf_id;
	char sen_manuf[8];
	char sen_type[16];
	u16 mod_id;
	u16 mod_rev;
	char regs[56];
	u16 nr_modes;
	u16 bytes_per_mode;
	char mode1[16];
	char mode2[16];
};

struct imx290 {
	struct v4l2_subdev subdev;
	struct media_pad pad;
	struct v4l2_ctrl_handler ctrl_handler;
	struct clk *clk;
	int hflip;
	int vflip;
	u16 analog_gain;
	u16 digital_gain;
	u32 exposure_time;
	struct v4l2_ctrl *pixel_rate;
	const struct imx290_mode *cur_mode;
	struct i2c_client *rom;
	struct vc_rom_table rom_table;
	struct mutex mutex;
	int streaming;
};

static const struct imx290_mode supported_modes[] = {
	{
		.sensor_mode = 0,
		.sensor_ext_trig = 0,
		.sensor_depth = 10,
		.width = 1920,
		.height = 1080,
		.max_fps = 60,
		.reg_list = imx290_init_tab_1920_1080_60fps,
	},
	{
		.sensor_mode = 1,
		.sensor_ext_trig = 0,
		.sensor_depth = 10,
		.width = 1920,
		.height = 1080,
		.max_fps = 120,
		.reg_list = imx290_init_tab_1920_1080_120fps,
	},
};

static struct imx290 *to_imx290(const struct i2c_client *client)
{
	return container_of(i2c_get_clientdata(client), struct imx290, subdev);
}

static int reg_write(struct i2c_client *client, const u16 addr, const u8 data)
{
	struct i2c_adapter *adap = client->adapter;
	struct i2c_msg msg;
	u8 tx[3];
	int ret;

	msg.addr = client->addr;
	msg.buf = tx;
	msg.len = 3;
	msg.flags = 0;
	tx[0] = addr >> 8;
	tx[1] = addr & 0xff;
	tx[2] = data;
	ret = i2c_transfer(adap, &msg, 1);
	mdelay(2);

	return ret == 1 ? 0 : -EIO;
}

static int reg_read(struct i2c_client *client, const u16 addr)
{
	u8 buf[2] = {addr >> 8, addr & 0xff};
	int ret;
	struct i2c_msg msgs[] = {
		{
			.addr  = client->addr,
			.flags = 0,
			.len   = 2,
			.buf   = buf,
		}, {
			.addr  = client->addr,
			.flags = I2C_M_RD,
			.len   = 1,
			.buf   = buf,
		},
	};

	ret = i2c_transfer(client->adapter, msgs, ARRAY_SIZE(msgs));
	if (ret < 0) {
		dev_warn(&client->dev, "Reading register %x from %x failed\n",
			 addr, client->addr);
		return ret;
	}

	return buf[0];
}

static int reg_write_table(struct i2c_client *client,
			   const struct imx290_reg table[])
{
	const struct imx290_reg *reg;
	int ret;

	for (reg = table; reg->addr != IMX290_TABLE_END; reg++) {
		ret = reg_write(client, reg->addr, reg->val);
		if (ret < 0)
			return ret;
	}

	return 0;
}

/* V4L2 subdev video operations */
static int imx290_s_stream(struct v4l2_subdev *sd, int enable)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct imx290 *priv = to_imx290(client);
	int ret;

	if (!enable)
		return reg_write_table(client, stop);

	ret = reg_write_table(client, priv->cur_mode->reg_list);
	if (ret)
		return ret;
	
	if(priv->cur_mode->sensor_ext_trig)
	{
		u64 exposure = (priv->exposure_time * 10000) / 185185;

		int addr = 0x0108; // ext trig enable
		int data =      1; // external trigger enable
		ret = reg_write(priv->rom, addr, data);
#if 0
		// TODO
		addr = 0x0103; // flash output enable
		data =      1; // flash output enable
		ret = reg_write(priv->rom, addr, data);
#endif
		addr = 0x0109; // shutter lsb
		data = exposure & 0xff;
		ret = reg_write(priv->rom, addr, data);

		addr = 0x010a;
		data = (exposure >> 8) & 0xff;
		ret = reg_write(priv->rom, addr, data);

		addr = 0x010b;
		data = (exposure >> 16) & 0xff;
		ret = reg_write(priv->rom, addr, data);

		addr = 0x010c; // shutter msb
		data = (exposure >> 24) & 0xff;
		ret = reg_write(priv->rom, addr, data);

		return reg_write_table(client, start);
	}	
	else
	{
		int addr = 0x0108; // ext trig enable
	       	int data =      0; // external trigger disable 
	        ret = reg_write(priv->rom, addr, data);
		
		return reg_write_table(client, start);
	}
}

/* V4L2 subdev core operations */
static int imx290_s_power(struct v4l2_subdev *sd, int on)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct imx290 *priv = to_imx290(client);

	mutex_lock(&priv->mutex);

	if (on)	{
		dev_dbg(&client->dev, "imx290 power on\n");
		clk_prepare_enable(priv->clk);
	} else if (!on) {
		dev_dbg(&client->dev, "imx290 power off\n");
		clk_disable_unprepare(priv->clk);
	}

	mutex_unlock(&priv->mutex);

	return 0;
}

static int imx290_s_ctrl(struct v4l2_ctrl *ctrl)
{
	struct imx290 *priv =
	    container_of(ctrl->handler, struct imx290, ctrl_handler);
	struct i2c_client *client = v4l2_get_subdevdata(&priv->subdev);
	u8 reg;
	int ret;
	u16 gain = 0;
	u32 exposure = 0;
	
	mutex_lock(&priv->mutex);

	switch (ctrl->id) {
	case V4L2_CID_HFLIP:
		priv->hflip = ctrl->val;
		break;

	case V4L2_CID_VFLIP:
		priv->vflip = ctrl->val;
		break;

	case V4L2_CID_GAIN:
		/*
		 * hal transfer (gain * 10)  to kernel
		 */

		gain = ctrl->val;

		if (gain < 0) 
			gain = 0;
		if (gain > 240)
			gain = 240;
		
		priv->digital_gain = gain;

		ret = reg_write(client, 0x3014, priv->digital_gain & 0xff);

		mutex_unlock(&priv->mutex);
		return ret;

	case V4L2_CID_EXPOSURE:


		priv->exposure_time = ctrl->val;
		
		if (priv->exposure_time < 29)
			priv->exposure_time = 29;
		if (priv->exposure_time > 7767184)
			priv->exposure_time = 7767184;

		if (priv->exposure_time < 38716)
		{
			// range 1..1123
			unsigned int lf = sensor_mode ? 2 : 1;
			exposure = (1124 * 20000 -  (int)(priv->exposure_time) * 29 * 20 * lf) / 20000;
			dev_info(&client->dev, "SHS = %d \n",exposure);

			ret  = reg_write(client, 0x301A, 0x00);  
			ret |= reg_write(client, 0x3019, 0x04);
			ret |= reg_write(client, 0x3018, 0x65);

			ret |= reg_write(client, 0x3022, (exposure >> 16) & 0x01);
			ret |= reg_write(client, 0x3021, (exposure >>  8) & 0xff);
			ret |= reg_write(client, 0x3020,  exposure        & 0xff);
		}
		else
		{
			// range 1123.. 
			unsigned int lf = sensor_mode ? 2 : 1;
			exposure = ( 1 * 20000 + (int)(priv->exposure_time) * 29 * 20 * lf ) / 20000;

			dev_info(&client->dev, "VMAX = %d \n",exposure);

			ret  = reg_write(client, 0x3022, 0x00);
			ret |= reg_write(client, 0x3021, 0x00);
			ret |= reg_write(client, 0x3020, 0x01);

			ret |= reg_write(client, 0x301A, (exposure >> 16) & 0x01);
			ret |= reg_write(client, 0x3019, (exposure >>  8) & 0xff);
			ret |= reg_write(client, 0x3018,  exposure        & 0xff);

		}

		mutex_unlock(&priv->mutex);
		return ret;

	default:
		mutex_unlock(&priv->mutex);
		return -EINVAL;
	}

	/* If enabled, apply settings immediately */
	reg = reg_read(client, 0x3000);
	if ((reg & 0x1f) == 0x01)
		imx290_s_stream(&priv->subdev, 1);

	mutex_unlock(&priv->mutex);

	return 0;
}

static int imx290_enum_mbus_code(struct v4l2_subdev *sd,
				 struct v4l2_subdev_pad_config *cfg,
				 struct v4l2_subdev_mbus_code_enum *code)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct imx290 *priv = to_imx290(client);
	const struct imx290_mode *mode = priv->cur_mode;

	if (code->index !=0)
		return -EINVAL;
	
	if(mode->sensor_depth==8)
		code->code = MEDIA_BUS_FMT_Y8_1X8;

	if(mode->sensor_depth==10)
		code->code = MEDIA_BUS_FMT_Y10_1X10;

	return 0;
}

#if 0
static int imx290_get_reso_dist(const struct imx290_mode *mode,
				struct v4l2_mbus_framefmt *framefmt)
{
	return abs(mode->width - framefmt->width) +
	       abs(mode->height - framefmt->height);
}
#endif

static const struct imx290_mode *imx290_find_best_fit(
					struct v4l2_subdev_format *fmt)
{
#if 0
	struct v4l2_mbus_framefmt *framefmt = &fmt->format;
	int dist;
	int cur_best_fit = 0;
	int cur_best_fit_dist = -1;
	int i;

	for (i = 0; i < ARRAY_SIZE(supported_modes); i++) {
		dist = imx290_get_reso_dist(&supported_modes[i], framefmt);
		if (cur_best_fit_dist == -1 || dist < cur_best_fit_dist) {
			cur_best_fit_dist = dist;
			cur_best_fit = i;
		}
	}

	return &supported_modes[cur_best_fit];
#else
	return &supported_modes[sensor_mode];
#endif
}

static int imx290_set_fmt(struct v4l2_subdev *sd,
			  struct v4l2_subdev_pad_config *cfg,
			  struct v4l2_subdev_format *fmt)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct imx290 *priv = to_imx290(client);
	const struct imx290_mode *mode;

	if (fmt->which == V4L2_SUBDEV_FORMAT_TRY)
		return 0;

	mode = imx290_find_best_fit(fmt);
	fmt->format.code = MEDIA_BUS_FMT_Y10_1X10;
	fmt->format.width = mode->width;
	fmt->format.height = mode->height;
	fmt->format.field = V4L2_FIELD_NONE;
	priv->cur_mode = mode;

	return 0;
}

static int imx290_get_fmt(struct v4l2_subdev *sd,
			  struct v4l2_subdev_pad_config *cfg,
			  struct v4l2_subdev_format *fmt)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct imx290 *priv = to_imx290(client);
	const struct imx290_mode *mode = priv->cur_mode;

	if (fmt->which == V4L2_SUBDEV_FORMAT_TRY)
		return 0;

	fmt->format.width = mode->width;
	fmt->format.height = mode->height;
	fmt->format.code = MEDIA_BUS_FMT_Y10_1X10;
	fmt->format.field = V4L2_FIELD_NONE;

	return 0;
}

/* Various V4L2 operations tables */
static struct v4l2_subdev_video_ops imx290_subdev_video_ops = {
	.s_stream = imx290_s_stream,
};

static struct v4l2_subdev_core_ops imx290_subdev_core_ops = {
	.s_power = imx290_s_power,
};

static const struct v4l2_subdev_pad_ops imx290_subdev_pad_ops = {
	.enum_mbus_code = imx290_enum_mbus_code,
	.set_fmt = imx290_set_fmt,
	.get_fmt = imx290_get_fmt,
};

static struct v4l2_subdev_ops imx290_subdev_ops = {
	.core = &imx290_subdev_core_ops,
	.video = &imx290_subdev_video_ops,
	.pad = &imx290_subdev_pad_ops,
};

static const struct v4l2_ctrl_ops imx290_ctrl_ops = {
	.s_ctrl = imx290_s_ctrl,
};

static int imx290_video_probe(struct i2c_client *client)
{
	u16 model_id;
	u32 lot_id=0;
	u16 chip_id=0;
	int ret;

	/* Check and show model, lot, and chip ID. */
	ret = reg_read(client, 0x359b);
	if (ret < 0) {
		dev_err(&client->dev, "Failure to read Model ID (high byte)\n");
		goto done;
	}
	model_id = ret << 8; 

	ret = reg_read(client, 0x359a);
	if (ret < 0) {
		dev_err(&client->dev, "Failure to read Model ID (low byte)\n");
		goto done;
	}
	model_id |= ret;
#if 0
	if ( ! ((model_id == 296) || (model_id == 202)) ) {
		dev_err(&client->dev, "Model ID: %x not supported!\n",
			model_id);
		ret = -ENODEV;
		goto done;
	}
#endif
	dev_info(&client->dev,
		 "Model ID 0x%04x, Lot ID 0x%06x, Chip ID 0x%04x\n",
		 model_id, lot_id, chip_id);
done:
	return ret;
}

static int imx290_ctrls_init(struct v4l2_subdev *sd)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);
	struct imx290 *priv = to_imx290(client);
	int ret;

	v4l2_ctrl_handler_init(&priv->ctrl_handler, 10);

	v4l2_ctrl_new_std(&priv->ctrl_handler, &imx290_ctrl_ops,
			  V4L2_CID_GAIN,
			  IMX290_DIGITAL_GAIN_MIN,
			  IMX290_DIGITAL_GAIN_MAX, 1,
			  IMX290_DIGITAL_GAIN_DEFAULT);

	v4l2_ctrl_new_std(&priv->ctrl_handler, &imx290_ctrl_ops,
			  V4L2_CID_EXPOSURE,
			  IMX290_DIGITAL_EXPOSURE_MIN,
			  IMX290_DIGITAL_EXPOSURE_MAX, 1,
			  IMX290_DIGITAL_EXPOSURE_DEFAULT);

	priv->subdev.ctrl_handler = &priv->ctrl_handler;
	if (priv->ctrl_handler.error) {
		dev_err(&client->dev, "Error %d adding controls\n",
			priv->ctrl_handler.error);
		ret = priv->ctrl_handler.error;
		goto error;
	}

	ret = v4l2_ctrl_handler_setup(&priv->ctrl_handler);
	if (ret < 0) {
		dev_err(&client->dev, "Error %d setting default controls\n",
			ret);
		goto error;
	}

	return 0;
error:
	v4l2_ctrl_handler_free(&priv->ctrl_handler);
	return ret;
}

static struct i2c_client * imx290_probe_vc_rom(struct i2c_adapter *adapter, u8 addr)
{
	struct i2c_board_info info = {
		I2C_BOARD_INFO("dummy", addr),
	};
	unsigned short addr_list[2] = { addr, I2C_CLIENT_END };

	return i2c_new_probed_device(adapter, &info, addr_list, NULL);
}

static int imx290_probe(struct i2c_client *client,
			const struct i2c_device_id *did)
{
	struct imx290 *priv;
	struct i2c_adapter *adapter = to_i2c_adapter(client->dev.parent);
	int ret;
#if 1
	struct device_node *endpoint;

	endpoint = of_graph_get_next_endpoint((&client->dev)->of_node, NULL);
	if (!endpoint) {
		dev_err(&client->dev, "endpoint node not found\n");
		return -EINVAL;
	}

	ret = fwnode_property_read_u32_array(of_fwnode_handle(endpoint), "data-lanes", NULL, 0);

	if(ret == 2)
		sensor_mode=0; // autoswitch to sensor_mode=0 if 2 lanes are given
	else if(ret == 4)
		sensor_mode=1; // autoswitch to sensor_mode=1 if 4 lanes are given
	else
	{
		dev_err(&client->dev, "VC Sensor device-tree: Invalid number of data-lanes found\n");
		return -EINVAL;
	}
	dev_info(&client->dev, "VC Sensor device-tree has configured %d data-lanes! [ sensor_mode = %d ]\n",ret,sensor_mode);
#endif

	if (!i2c_check_functionality(adapter, I2C_FUNC_SMBUS_BYTE_DATA)) {
		dev_warn(&adapter->dev,
			 "I2C-Adapter doesn't support I2C_FUNC_SMBUS_BYTE\n");
		return -EIO;
	}
	priv = devm_kzalloc(&client->dev, sizeof(struct imx290), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;
	
	mutex_init(&priv->mutex);

	priv->clk = devm_clk_get(&client->dev, NULL);
	if (IS_ERR(priv->clk)) {
		dev_info(&client->dev, "Error %ld getting clock\n",
			 PTR_ERR(priv->clk));
		return -EPROBE_DEFER;
	}

	priv->cur_mode = &supported_modes[sensor_mode];

	priv->rom = imx290_probe_vc_rom(adapter,0x10);

 	if ( priv->rom )
	{	
		static int i=1;
		int addr,reg,data;
#if 1
		for (addr=0; addr<sizeof(priv->rom_table); addr++)
		{
	          reg = reg_read(priv->rom, addr+0x1000);
		  if(reg < 0)
		  {
			  i2c_unregister_device(priv->rom);
			  return -EIO;
		  }
		  *((char *)(&(priv->rom_table))+addr)=(char)reg;
		  dev_dbg(&client->dev, "addr=0x%04x reg=0x%02x\n",addr+0x1000,reg);
		}

		dev_info(&client->dev, "VC FPGA found!\n");

		dev_info(&client->dev, "[ MAGIC  ] [ %s ]\n",
				priv->rom_table.magic);

		dev_info(&client->dev, "[ MANUF. ] [ %s ] [ MID=0x%04x ]\n",
				priv->rom_table.manuf,
				priv->rom_table.manuf_id);

		dev_info(&client->dev, "[ SENSOR ] [ %s %s ]\n",
				priv->rom_table.sen_manuf,
				priv->rom_table.sen_type);

		dev_info(&client->dev, "[ MODULE ] [ ID=0x%04x ] [ REV=0x%04x ]\n",
				priv->rom_table.mod_id,
				priv->rom_table.mod_rev);

		dev_info(&client->dev, "[ MODES  ] [ NR=0x%04x ] [ BPM=0x%04x ]\n",
				priv->rom_table.nr_modes,
				priv->rom_table.bytes_per_mode);
#endif
		addr = 0x0100; // reset
	       	data =      2; // powerdown sensor 
	        reg = reg_write(priv->rom, addr, data);
		
		addr = 0x0102; // mode
	       	data = sensor_mode; // default 10-bit streaming
	        reg = reg_write(priv->rom, addr, data);
			
		addr = 0x0100; // reset
	       	data =      0; // powerup sensor
	        reg = reg_write(priv->rom, addr, data);
		
		while(1)
		{
			mdelay(100); // wait 100ms 
			
			addr = 0x0101; // status
			reg = reg_read(priv->rom, addr);
			
			if(reg & 0x80)
			       	break;
			
			if(reg & 0x01) 	
				dev_err(&client->dev, "!!! ERROR !!! setting VC Sensor MODE=%d STATUS=0x%02x i=%d\n",sensor_mode,reg,i);
			
			if(i++ >  4)
				break;
		}

		dev_info(&client->dev, "VC Sensor MODE=%d PowerOn STATUS=0x%02x i=%d\n",sensor_mode,reg,i);

	}
	else
	{
		dev_err(&client->dev, "Error !!! VC FPGA not found !!!\n");
		return -EIO;
	}

	ret = imx290_video_probe(client);
	if (ret < 0)
	{
		ret = -EIO;
		goto err_out;
	}

	v4l2_i2c_subdev_init(&priv->subdev, client, &imx290_subdev_ops);
	ret = imx290_ctrls_init(&priv->subdev);
	if (ret < 0)
		goto err_out;

	priv->subdev.flags |= V4L2_SUBDEV_FL_HAS_DEVNODE;
	priv->pad.flags = MEDIA_PAD_FL_SOURCE;

	priv->subdev.entity.function = MEDIA_ENT_F_CAM_SENSOR;
	ret = media_entity_pads_init(&priv->subdev.entity, 1, &priv->pad);
	if (ret < 0)
		goto err_out;

	ret = v4l2_async_register_subdev(&priv->subdev);

err_out:	
	if (ret < 0)
	{
		if(priv->rom)
			i2c_unregister_device(priv->rom);
		return ret;
	}

	return ret;
}

static int imx290_remove(struct i2c_client *client)
{
	struct imx290 *priv = to_imx290(client);

	if(priv->rom)
		i2c_unregister_device(priv->rom);
	v4l2_async_unregister_subdev(&priv->subdev);
	media_entity_cleanup(&priv->subdev.entity);
	v4l2_ctrl_handler_free(&priv->ctrl_handler);

	return 0;
}

static const struct i2c_device_id imx290_id[] = {
	{"vc_mipi_imx290", 0},
	{}
};

static const struct of_device_id imx290_of_match[] = {
	{ .compatible = "sony,vc_mipi_imx290" },
	{ /* sentinel */ },
};
MODULE_DEVICE_TABLE(of, imx290_of_match);

MODULE_DEVICE_TABLE(i2c, imx290_id);
static struct i2c_driver imx290_i2c_driver = {
	.driver = {
		.of_match_table = of_match_ptr(imx290_of_match),
		.name = "vc_mipi_imx290",
	},
	.probe = imx290_probe,
	.remove = imx290_remove,
	.id_table = imx290_id,
};

module_i2c_driver(imx290_i2c_driver);
MODULE_VERSION("0.0.4");
MODULE_DESCRIPTION("Vision Components GmbH - VC MIPI IMX290 driver");
MODULE_AUTHOR("Michael Steinel, Vision Components GmbH <mipi-tech@vision-components.com>");
MODULE_LICENSE("GPL v2");
